﻿// ---------------------------------------------------------------
// <copyright file="DialogCenteringService.cs" company="B33Rbaron">
//     Copyright (c) Daniel Birler.
//     Licensed under Microsoft Public License (Ms-PL).
// </copyright>
// <author>Daniel Birler</author>
// ---------------------------------------------------------------

namespace ChromeHost
{
  using System;
  using System.Drawing;
  using System.Text;
  using System.Windows.Forms;

  /// <summary>
  /// A class for centering windows dialogs.
  /// </summary>
  public sealed class DialogCenteringService : IDisposable
  {
    /// <summary>
    /// The number of how often it was tried to find a windows dialog.
    /// </summary>
    private int numberOfTries = 0;

    /// <summary>
    /// The form owning the windows dialog.
    /// </summary>
    private Form ownerForm = null;

    /// <summary>
    /// Initializes a new instance of the DialogCenteringService class.
    /// </summary>
    /// <param name="ownerForm">The form owning the windows dialog.</param>
    public DialogCenteringService(Form ownerForm)
    {
      this.ownerForm = ownerForm;
      this.ownerForm.BeginInvoke(new MethodInvoker(this.SearchDialogAndCenter));
    }

    /// <summary>
    /// Implementing from IDisposable.Dispose.
    /// </summary>
    public void Dispose()
    {
      this.numberOfTries = -1;
    }

    /// <summary>
    /// Searches the current thread for a windows dialog and centers it on the owning form.
    /// </summary>
    private void SearchDialogAndCenter()
    {
      if (this.numberOfTries >= 0)
      {
        this.numberOfTries++;
        NativeMethods.EnumThreadWndProc callback = new NativeMethods.EnumThreadWndProc(this.EnumThreadWndProcCallback);
        if (NativeMethods.EnumThreadWindows(NativeMethods.GetCurrentThreadId(), callback, IntPtr.Zero))
        {
          if (this.numberOfTries < 10)
          {
            this.ownerForm.BeginInvoke(new MethodInvoker(this.SearchDialogAndCenter));
          }
        }
      }
    }

    /// <summary>
    /// An application-defined callback function used with the EnumThreadWindows function. It
    /// receives the window handles associated with a thread. The WNDENUMPROC type defines a
    /// pointer to this callback function. EnumThreadWndProc is a placeholder for the
    /// application-defined function name.
    /// </summary>
    /// <param name="windowHandle">A handle to a window associated with the thread specified in the
    /// EnumThreadWindows function.</param>
    /// <param name="longParam">The application-defined value given in the EnumThreadWindows function.</param>
    /// <returns>To continue enumeration, the callback function must return TRUE;
    /// to stop enumeration, it must return FALSE.</returns>
    private bool EnumThreadWndProcCallback(IntPtr windowHandle, IntPtr longParam)
    {
      StringBuilder stringBuilder = new StringBuilder(260);
      int characterCount = NativeMethods.GetClassName(windowHandle, stringBuilder, stringBuilder.Capacity);

      if (characterCount != 0 && stringBuilder.ToString() == "#32770")
      {
        Rectangle ownerFormDesktopBounds = this.ownerForm.DesktopBounds;
        NativeMethods.RECT windowRectangle;
        NativeMethods.GetWindowRect(windowHandle, out windowRectangle);

        int left = ownerFormDesktopBounds.Left + ((ownerFormDesktopBounds.Width - windowRectangle.Right + windowRectangle.Left) / 2);
        int top = ownerFormDesktopBounds.Top + ((ownerFormDesktopBounds.Height - windowRectangle.Bottom + windowRectangle.Top) / 2);
        int width = windowRectangle.Right - windowRectangle.Left;
        int height = windowRectangle.Bottom - windowRectangle.Top;
        NativeMethods.MoveWindow(windowHandle, left, top, width, height, true);

        return false;
      }

      return true;
    }
  }
}