﻿// ---------------------------------------------------------------
// <copyright file="Program.cs" company="B33Rbaron">
//     Copyright (c) Daniel Birler.
//     Licensed under Microsoft Public License (Ms-PL).
// </copyright>
// <author>Daniel Birler</author>
// ---------------------------------------------------------------

namespace ChromeHost
{
  using System;
  using System.Collections.Generic;
  using System.Windows.Forms;
  using ChromeHost.Properties;
  using Microsoft.Win32;

  /// <summary>
  /// The applications program class.
  /// </summary>
  public static class Program
  {
    /// <summary>
    /// The notify icon.
    /// </summary>
    private static NotifyIcon notifyIcon = null;

    /// <summary>
    /// Stores the menu items created from the settings.
    /// </summary>
    private static List<MenuItemEx> menuItems = null;

    /// <summary>
    /// The settings context menu item.
    /// </summary>
    private static MenuItemEx settingsMenuItem = null;

    /// <summary>
    /// The about context menu item.
    /// </summary>
    private static MenuItemEx aboutMenuItem = null;

    /// <summary>
    /// The exit context menu item.
    /// </summary>
    private static MenuItemEx exitMenuItem = null;

    /// <summary>
    /// The context menu of the notify icon.
    /// </summary>
    private static ContextMenuEx contextMenu = null;

    /// <summary>
    /// The window receiving the messages.
    /// </summary>
    private static DetectorWindow detectorWindow = null;

    /// <summary>
    /// The main entry point for the application.
    /// </summary>
    [STAThread]
    public static void Main()
    {
      Application.EnableVisualStyles();
      Application.SetCompatibleTextRenderingDefault(false);

      if (!ChromeHelper.Instance.IsChromeInstalled())
      {
        MessageBox.Show("No installation of Google Chrome could be located on your device. ChromeHost will not work without Google Chrome and must exit now.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error, MessageBoxDefaultButton.Button1, 0);
        return;
      }

      SystemEvents.SessionEnding += SystemEventsSessionEnding;

      Settings.CheckForUpgrade();

      Program.menuItems = new List<MenuItemEx>();
      Program.settingsMenuItem = new MenuItemEx("Settings...", Resources.MenuSettings, Program.MenuItemSettingsClick);
      Program.aboutMenuItem = new MenuItemEx("About...", Resources.MenuAbout, Program.MenuItemAboutClick);
      Program.exitMenuItem = new MenuItemEx("Exit", Resources.MenuExit, Program.MenuItemExitClick);

      Program.contextMenu = new ContextMenuEx();
      Program.contextMenu.MenuItems.Add(Program.settingsMenuItem);
      Program.contextMenu.MenuItems.Add(Program.aboutMenuItem);
      Program.contextMenu.MenuItems.Add(new MenuItemEx("-"));
      Program.contextMenu.MenuItems.Add(Program.exitMenuItem);
      Program.contextMenu.Popup += Program.ContextMenuPopup;

      Program.notifyIcon = new NotifyIcon();
      Program.notifyIcon.Icon = Properties.Resources.ChromeHost;
      Program.notifyIcon.Visible = true;
      Program.notifyIcon.ContextMenu = Program.contextMenu;

      if (Settings.Default.ShowInfoOnStart)
      {
        FormManager.Instance.ShowInfoForm();
      }

      Program.detectorWindow = new DetectorWindow();
      Application.Run();
    }

    /// <summary>
    /// Occurs before the shortcut menu is displayed.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">A System.EventArgs that contains the event data.</param>
    private static void ContextMenuPopup(object sender, EventArgs e)
    {
      int index = 0;
      Program.PreCleanContextMenu();
      Program.menuItems.AddRange(NameUrlPairManager.Instance.RetrieveMenuItems());

      foreach (MenuItemEx menuItem in Program.menuItems)
      {
        menuItem.Click += Program.MenuItemClick;
        Program.contextMenu.MenuItems.Add(index++, menuItem);
      }

      if (Program.menuItems.Count > 0)
      {
        Program.contextMenu.MenuItems.Add(index, new MenuItemEx("-"));
      }

      Program.contextMenu.CreateMenuItemImages();
    }

    /// <summary>
    /// Occurs when a dynamic menu item from the settings is clicked.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">A System.EventArgs that contains the event data.</param>
    private static void MenuItemClick(object sender, EventArgs e)
    {
      MenuItemEx menuItemSender = sender as MenuItemEx;
      if (menuItemSender != null)
      {
        NameUrlPair nameUrlPairSender = menuItemSender.Tag as NameUrlPair;
        if (nameUrlPairSender != null)
        {
          ChromeHelper.Instance.RunAsApp(nameUrlPairSender.Url);
        }
      }
    }

    /// <summary>
    /// Occurs when the settings menu item is clicked.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">A System.EventArgs that contains the event data.</param>
    private static void MenuItemSettingsClick(object sender, EventArgs e)
    {
      FormManager.Instance.ShowSettingsForm();
    }

    /// <summary>
    /// Occurs when the about menu item is clicked.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">A System.EventArgs that contains the event data.</param>
    private static void MenuItemAboutClick(object sender, EventArgs e)
    {
      FormManager.Instance.ShowAboutForm();
    }

    /// <summary>
    /// Occurs when the exit menu item is clicked.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">A System.EventArgs that contains the event data.</param>
    private static void MenuItemExitClick(object sender, EventArgs e)
    {
      SystemEvents.SessionEnding -= SystemEventsSessionEnding;

      Program.settingsMenuItem.Click -= Program.MenuItemSettingsClick;
      Program.aboutMenuItem.Click -= Program.MenuItemAboutClick;
      Program.exitMenuItem.Click -= Program.MenuItemExitClick;

      Program.PreCleanContextMenu();
      Program.contextMenu.MenuItems.Clear();
      Program.contextMenu.Popup -= Program.ContextMenuPopup;
      Program.contextMenu.Dispose();

      Program.notifyIcon.ContextMenu = null;
      Program.notifyIcon.Visible = false;
      Program.notifyIcon.Dispose();

      Program.detectorWindow.Dispose();

      Application.Exit();
    }

    /// <summary>
    /// Cleans the context menu from entries created from the settings.
    /// Clears the list that stores the menu items created from the settings.
    /// </summary>
    private static void PreCleanContextMenu()
    {
      foreach (MenuItemEx menuItem in Program.menuItems)
      {
        menuItem.Click -= Program.MenuItemClick;
        Program.contextMenu.MenuItems.Remove(menuItem);
        menuItem.Dispose();
      }

      if (Program.menuItems.Count > 0)
      {
        MenuItemEx menuItem = (MenuItemEx)Program.contextMenu.MenuItems[0];
        Program.contextMenu.MenuItems.Remove(menuItem);
        menuItem.Dispose();
        Program.menuItems.Clear();
      }
    }

    /// <summary>
    /// Occurs when the user is logging off or shutting down the system.
    /// </summary>
    /// <param name="sender">The source of the  event.</param>
    /// <param name="e">A Microsoft.Win32.SessionEndingEventArgs that contains the event data.</param>
    private static void SystemEventsSessionEnding(object sender, SessionEndingEventArgs e)
    {
      Program.MenuItemExitClick(null, EventArgs.Empty);
    }
  }
}