﻿// ---------------------------------------------------------------
// <copyright file="ChromeWindowWatcher.cs" company="B33Rbaron">
//     Copyright (c) Daniel Birler.
//     Licensed under Microsoft Public License (Ms-PL).
// </copyright>
// <author>Daniel Birler</author>
// ---------------------------------------------------------------

namespace ChromeHost
{
  using System;
  using System.Collections.Generic;
  using System.Linq;
  using System.Runtime.InteropServices;
  using System.Text;

  /// <summary>
  /// A class for watching for Chrome windows to be created and shown.
  /// </summary>
  internal sealed class ChromeWindowWatcher
  {
    /// <summary>
    /// This field is responsible for initializing the ChromeWindowWatcher class.
    /// </summary>
    private static readonly Lazy<ChromeWindowWatcher> Lazy = new Lazy<ChromeWindowWatcher>(() => new ChromeWindowWatcher());

    /// <summary>
    /// Indicates whether the event procedure is enabled or not.
    /// </summary>
    private bool enabled = false;

    /// <summary>
    /// The event procedure.
    /// </summary>
    private NativeMethods.WinEventProc winEventProc = null;

    /// <summary>
    /// The handle of the event hook.
    /// </summary>
    private IntPtr eventHookHandle = IntPtr.Zero;

    /// <summary>
    /// The garbage collected handle for the event procedure.
    /// </summary>
    private GCHandle garbageCollectedHandle;

    /// <summary>
    /// Prevents a default instance of the ChromeWindowWatcher class from being created.
    /// </summary>
    private ChromeWindowWatcher()
    {
      this.winEventProc = new NativeMethods.WinEventProc(this.WinEventProc);
      this.garbageCollectedHandle = GCHandle.Alloc(this.winEventProc);
    }

    /// <summary>
    /// This event handler is fired when a new Chrome window is shown.
    /// </summary>
    public event EventHandler<WindowCreatedEventArgs> WindowCreated;

    /// <summary>
    /// Gets the one and only instance of the ChromeWindowWatcher class.
    /// </summary>
    public static ChromeWindowWatcher Instance
    {
      get { return Lazy.Value; }
    }

    /// <summary>
    /// Gets or sets a value indicating whether the event procedure is enabled or not.
    /// </summary>
    public bool Enabled
    {
      get
      {
        return this.enabled;
      }

      set
      {
        this.enabled = value;
        this.UpdateWinEventHook();
      }
    }

    /// <summary>
    /// Responsible for updating the event hook.
    /// </summary>
    private void UpdateWinEventHook()
    {
      if (this.eventHookHandle != IntPtr.Zero)
      {
        NativeMethods.UnhookWinEvent(this.eventHookHandle);
        this.eventHookHandle = IntPtr.Zero;
      }

      if (this.enabled)
      {
        this.eventHookHandle = NativeMethods.SetWinEventHook(
          NativeMethods.EVENT_OBJECT_SHOW,
          NativeMethods.EVENT_OBJECT_SHOW,
          IntPtr.Zero,
          this.winEventProc,
          0,
          0,
          NativeMethods.WINEVENT_OUTOFCONTEXT | NativeMethods.WINEVENT_SKIPOWNPROCESS);
      }
    }

    /// <summary>
    /// An application-defined callback (or hook) function that the system calls in response to
    /// events generated by an accessible object. The hook function processes the event notifications
    /// as required. Clients install the hook function and request specific types of event notifications
    /// by calling SetWinEventHook.
    /// </summary>
    /// <param name="winEventHookHandle">Handle to an event hook function. This value is returned by
    /// SetWinEventHook when the hook function is installed and is specific to each instance of
    /// the hook function.</param>
    /// <param name="eventType">Specifies the event that occurred.</param>
    /// <param name="windowHandle">Handle to the window that generates the event, or NULL if no window
    /// is associated with the event. For example, the mouse pointer is not associated with a window.</param>
    /// <param name="idObject">Identifies the object associated with the event.</param>
    /// <param name="idChild">Identifies whether the event was triggered by an object or a child element
    /// of the object. If this value is CHILDID_SELF, the event was triggered by the object; otherwise,
    /// this value is the child ID of the element that triggered the event.</param>
    /// <param name="idEventThread">Identifies the thread that generated the event, or the thread
    /// that owns the current window.</param>
    /// <param name="dwmsEventTime">Specifies the time, in milliseconds, that the event was generated.</param>
    private void WinEventProc(
      IntPtr winEventHookHandle,
      uint eventType,
      IntPtr windowHandle,
      long idObject,
      long idChild,
      uint idEventThread,
      uint dwmsEventTime)
    {
      if (winEventHookHandle == this.eventHookHandle && eventType == NativeMethods.EVENT_OBJECT_SHOW)
      {
        if (idObject == NativeMethods.OBJID_WINDOW /*&& idChild == NativeMethods.CHILDID_SELF*/)
        {
          System.Text.StringBuilder stringBuilder = new System.Text.StringBuilder(260);
          int characterCount = NativeMethods.GetClassName(windowHandle, stringBuilder, stringBuilder.Capacity);

          if (characterCount != 0 && stringBuilder.ToString() == "Chrome_WidgetWin_1")
          {
            if (this.WindowCreated != null)
            {
              this.WindowCreated(this, new WindowCreatedEventArgs(windowHandle));
            }
          }
        }
      }
    }

    /// <summary>
    /// A class for containing window created event data.
    /// </summary>
    internal class WindowCreatedEventArgs : EventArgs
    {
      /// <summary>
      /// The window handle from the window which was created.
      /// </summary>
      private IntPtr windowHandle;

      /// <summary>
      /// Initializes a new instance of the WindowCreatedEventArgs class.
      /// </summary>
      /// <param name="windowHandle">The window handle from the window which was created.</param>
      public WindowCreatedEventArgs(IntPtr windowHandle)
      {
        this.windowHandle = windowHandle;
      }

      /// <summary>
      /// Gets the window handle from the window which was created.
      /// </summary>
      public IntPtr WindowHandle
      {
        get { return this.windowHandle; }
      }
    }
  }
}