﻿// ---------------------------------------------------------------
// <copyright file="MenuItemEx.cs" company="B33Rbaron">
//     Copyright (c) Daniel Birler.
//     Licensed under Microsoft Public License (Ms-PL).
// </copyright>
// <author>Daniel Birler</author>
// ---------------------------------------------------------------

namespace ChromeHost
{
  using System;
  using System.Drawing;
  using System.Windows.Forms;

  /// <summary>
  /// An extended MenuItem class.
  /// </summary>
  public sealed class MenuItemEx : MenuItem
  {
    /// <summary>
    /// The image.
    /// </summary>
    private Bitmap image;

    /// <summary>
    /// The image handle.
    /// </summary>
    private IntPtr imageHandle;

    /// <summary>
    /// Initializes a new instance of the MenuItemEx class.
    /// </summary>
    /// <param name="text">The caption for the menu item.</param>
    public MenuItemEx(string text)
      : base(text)
    {
    }

    /// <summary>
    /// Initializes a new instance of the MenuItemEx class.
    /// </summary>
    /// <param name="text">The caption for the menu item.</param>
    /// <param name="image">The image for the menu item.</param>
    public MenuItemEx(string text, Bitmap image)
      : base(text)
    {
      this.image = image;
    }

    /// <summary>
    /// Initializes a new instance of the MenuItemEx class.
    /// </summary>
    /// <param name="text">The caption for the menu item.</param>
    /// <param name="image">The image for the menu item.</param>
    /// <param name="clickHandler">The System.EventHandler that handles the
    /// System.Windows.Forms.MenuItem.Click event for this menu item.</param>
    public MenuItemEx(string text, Bitmap image, EventHandler clickHandler)
      : base(text)
    {
      this.Click += clickHandler;
      this.image = image;
    }

    /// <summary>
    /// Adds the image to the menu item.
    /// </summary>
    /// <returns>True if successful, otherwise false.</returns>
    public bool AddImageToMenuItem()
    {
      if (this.image != null)
      {
        if (this.imageHandle == IntPtr.Zero)
        {
          this.imageHandle = this.image.GetHbitmap();
        }

        return NativeMethods.SetMenuItemBitmaps(this.Parent.Handle, (uint)this.MenuID, NativeMethods.MF_BYCOMMAND, this.imageHandle, this.imageHandle);
      }

      return true;
    }

    /// <summary>
    /// Releases the unmanaged resources used by the System.Windows.Forms.Control
    /// and its child controls and optionally releases the managed resources.
    /// </summary>
    /// <param name="disposing">True to release both managed and unmanaged resources;
    /// False to release only unmanaged resources.</param>
    protected override void Dispose(bool disposing)
    {
      if (disposing)
      {
        if (this.image != null)
        {
          this.image.Dispose();
        }
      }

      if (this.imageHandle != IntPtr.Zero)
      {
        NativeMethods.DeleteObject(this.imageHandle);
      }

      base.Dispose(disposing);
    }
  }
}